function [VelOut,goodrows] = ASCII2GIS(drange,ref)
% WinRiver ASCII to GIS Import Format 

% This program reads in an ASCII file or files generated from WinRiver 
% Classic ASCII output and outputs the Georeferenced mean velocity,
% temperature, depth, and backscatter data to a file for input into GIS.

% drange = [depth1 depth2] %range of depths over which to average the data
% ('dfs' option)
% (full range of data if blank)  %Added 12-20-10

% drange = single value for 'hab' option (height above bottom in m)

% ref = 'dfs' or 'hab';  %'dsf' = depth from surface; 'hab' = height above
% bottom

%Updated directional averaging PRJ 2/8/11
%Updated save path PRJ 3/10/11
%Added *.anv file export PRJ 5-11-11

% P.R. Jackson 6-25-10

%% USer inputs
append_data = 1;

%% Check the inputs

if nargin == 0
    drange = [];
    ref = 'dfs';
elseif nargin < 2
    ref = 'dfs';
end

%% Read and Convert the Data

% Determine Files to Process
% Prompt user for directory containing files
defaultpath = 'C:\';
ascii2gispath = [];
if exist('\VMT\LastDir.mat') == 2
    load('\VMT\LastDir.mat');
    if exist(ascii2gispath) == 7
        ascii2gispath = uigetdir(ascii2gispath,'Select the Directory Containing ASCII Output Data Files (*.txt)');
    else
        ascii2gispath = uigetdir(defaultpath,'Select the Directory Containing ASCII Output Data Files (*.txt)');
    end
else
    ascii2gispath = uigetdir(defaultpath,'Select the Directory Containing ASCII Output Data Files (*.txt)');
end
zPathName = ascii2gispath;
Files = dir(zPathName);
allFiles = {Files.name};
filefind = strfind(allFiles,'ASC.TXT')';
filesidx=nan(size(filefind,1),1);
for i=1:size(filefind,1)
    filesidx(i,1)=size(filefind{i},1);
end
filesidx=find(filesidx>0);
files=allFiles(filesidx);

if isempty(files)
    errordlg(['No ASC.TXT files found in ' ascii2gispath '.  Ensure data files are in the form "*_ASC.TXT" (Standard WRII naming convention).']);
end

% Allow user to select which files are to be processed
selection = listdlg('ListSize',[300 300],'ListString', files,'Name','Select Data Files');
zFileName = files(selection);

% Determine number of files to be processed
if  isa(zFileName,'cell')
    z=size(zFileName,2);
    zFileName=sort(zFileName);       
else
    z=1;
    zFileName={zFileName}
end
msgbox('Loading Data...Please Be Patient','Conversion Status','help','replace');
% Read in Selected Files
% Initialize row counter

% Begin master loop

VelOut = [];  %Matrix for output of velocity data

for zi=1:z
    %Clear variables
    clear DAVeast DAVnorth DAVmag DAVdir DAVvert ustar zo cod i j
    
    % Open txt data file
    if  isa(zFileName,'cell')
        fileName=strcat(zPathName,'\',zFileName(zi));
        fileName=char(fileName);
    else
        fileName=strcat(zPathName,zFileName);
    end

    % screenData 0 leaves bad data as -32768, 1 converts to NaN
    screenData=1;

    % tfile reads the data from an RDI ASCII output file and puts the
    % data in a Matlab data structure with major groups of:
    % Sup - supporing data
    % Wat - water data
    % Nav - navigation data including GPS.
    % Sensor - Sensor data
    % Q - discharge related data
    [A]=tfile(fileName,screenData);
    %Extract only Lat lon data
    latlon(:,1)=A.Nav.lat_deg(:,:);
    latlon(:,2)=A.Nav.long_deg(:,:);
    
    %Rescreen data to remove missing data (30000 value)
    indx1 = find(abs(latlon(:,1)) > 90);
    indx2 = find(abs(latlon(:,2)) > 180);
    latlon(indx1,1)=NaN;
    latlon(indx2,2)=NaN;
    
    indx3 = find(~isnan(latlon(:,1)) & ~isnan(latlon(:,2)));
    latlon = latlon(indx3,:);
    
    %Extract the Depths
    BeamDepths  = A.Nav.depth(indx3,:);
    Depth = nanmean(A.Nav.depth(indx3,:),2);
    
    %Filter Backscatter 
    A = VMT_FilterBS(1,A);
    
    
    %Extract the averaged velocities and backscatter (layer average)
    if isempty(drange)  
        disp(['Extracting DFS Range = Full'])
        DAVeast  = VMT_LayerAveMean(A.Wat.binDepth(:,indx3),A.Wat.vEast(:,indx3));
        DAVnorth = VMT_LayerAveMean(A.Wat.binDepth(:,indx3),A.Wat.vNorth(:,indx3));
        DAVvert  = VMT_LayerAveMean(A.Wat.binDepth(:,indx3),A.Wat.vVert(:,indx3));
        DABack   = VMT_LayerAveMean(A.Wat.binDepth(:,indx3),A.Clean.bsf(:,indx3));
        %DAVeast  = nanmean(A.Wat.vEast(:,indx3),1)';
        %DAVnorth = nanmean(A.Wat.vNorth(:,indx3),1)';
        %DAVvert  = nanmean(A.Wat.vVert(:,indx3),1)';
        %DABack   = nanmean(A.Clean.bsf(:,indx3),1)';
        DAVeast  = DAVeast';
        DAVnorth = DAVnorth';
        DAVvert  = DAVvert';
        DABack   = DABack';
    elseif strcmp(ref,'dfs')
        disp(['Extracting DFS Range = ' num2str(drange(1)) ' to ' num2str(drange(2)) ' m'])
        indxr = find(A.Wat.binDepth(:,1) >= drange(1) & A.Wat.binDepth(:,1) <= drange(2));
        DAVeast  = VMT_LayerAveMean(A.Wat.binDepth(indxr,indx3),A.Wat.vEast(indxr,indx3));
        DAVnorth = VMT_LayerAveMean(A.Wat.binDepth(indxr,indx3),A.Wat.vNorth(indxr,indx3));
        DAVvert  = VMT_LayerAveMean(A.Wat.binDepth(indxr,indx3),A.Wat.vVert(indxr,indx3));
        DABack   = VMT_LayerAveMean(A.Wat.binDepth(indxr,indx3),A.Clean.bsf(indxr,indx3));        
        %DAVeast  = nanmean(A.Wat.vEast(indxr,indx3),1)';
        %DAVnorth = nanmean(A.Wat.vNorth(indxr,indx3),1)';
        %DAVvert  = nanmean(A.Wat.vVert(indxr,indx3),1)';
        %DABack   = nanmean(A.Clean.bsf(indxr,indx3),1)';
        DAVeast  = DAVeast';
        DAVnorth = DAVnorth';
        DAVvert  = DAVvert';
        DABack   = DABack';
    elseif strcmp(ref,'hab')
        disp(['Extracting HAB Limit = ' num2str(drange) ' m'])
        i = 1;
        for j = 1:length(indx3)
            bed = nanmean(A.Nav.depth(indx3(j),:),2)';
            indxr = find(A.Wat.binDepth(:,1) >= (bed - drange(1)) & A.Wat.binDepth(:,1) <= bed);
%             DAVeast(i)  = VMT_LayerAveMean(A.Wat.binDepth(indxr,indx3(j)),A.Wat.vEast(indxr,indx3(j)));
%             DAVnorth(i) = VMT_LayerAveMean(A.Wat.binDepth(indxr,indx3(j)),A.Wat.vNorth(indxr,indx3(j)));
%             DAVvert(i)  = VMT_LayerAveMean(A.Wat.binDepth(indxr,indx3(j)),A.Wat.vVert(indxr,indx3(j)));
%             DABack(i)   = VMT_LayerAveMean(A.Wat.binDepth(indxr,indx3(j)),A.Clean.bsf(indxr,indx3(j)));
            DAVeast(i)  = nanmean(A.Wat.vEast(indxr,indx3(j)),1);
            DAVnorth(i) = nanmean(A.Wat.vNorth(indxr,indx3(j)),1);
            DAVvert(i)  = nanmean(A.Wat.vVert(indxr,indx3(j)),1);
            DABack(i)   = nanmean(A.Clean.bsf(indxr,indx3(j)),1)';
            
            i = i + 1;
        end
        
        DAVeast  = DAVeast';
        DAVnorth = DAVnorth';
        DAVvert  = DAVvert';
        DABack   = DABack';
    end
    
    % Compute the magnitude from the components
    DAVmag   = sqrt(DAVeast.^2 + DAVnorth.^2);
    
    % Compute the average direction from the velocity components
    DAVdir = 90 - (atan2(DAVnorth, DAVeast))*180/pi; %Compute the atan from the velocity componentes, convert to radians, and rotate to north axis
    qindx = find(DAVdir < 0);
    if ~isempty(qindx)
        DAVdir(qindx) = DAVdir(qindx) + 360;  %Must add 360 deg to Quadrant 4 values as they are negative angles from the +y axis
    end
        
    %Extract the Sensors
    Pitch = A.Sensor.pitch_deg(indx3);
    Roll  = A.Sensor.roll_deg(indx3);
    Heading  = A.Sensor.heading_deg(indx3);
    Temp  = A.Sensor.temp_degC(indx3);
    
    %Extract the time stamps
    MTyear      = A.Sup.year(indx3) + 2000; %works for data file after the year 2000
    MTmon       = A.Sup.month(indx3);
    MTday       = A.Sup.day(indx3);
    MThour      = A.Sup.hour(indx3);
    MTmin       = A.Sup.minute(indx3);
    MTsec       = A.Sup.second(indx3) + A.Sup.sec100(indx3)/100;

    %Extract Ens No
    EnsNo = A.Sup.ensNo(indx3);
    
    %Fit the profile to estimate the shear velocity
    if 0 %Compute the shear velocity through fittingof profiles
        clear i j
        i = 1;
        for j = 1:length(indx3)
            dfit = nanmean(A.Nav.depth(indx3(j),:),2);
            zfit = dfit - A.Wat.binDepth(:,1);
            znorm = zfit./dfit;
            indxfr = find(znorm >= 0.2 & znorm <= 1); %takes only data above 0.2H
            ufit = A.Wat.vMag(indxfr,indx3(j))/100;
            zfit = zfit(indxfr);
            indxgd = find(~isnan(ufit));
            if ~isempty(indxgd)
                [ustar(i),zo(i),cod(i)] = fitLogLawV2(ufit(indxgd),zfit(indxgd),dfit);
                if cod(i) < 0.25 | ustar(i) < 0 | zo(i) > 1.0  %screens the results
                    ustar(i) = nan;
                    zo(i) = nan;
                end
            else
                ustar(i) = nan;
                zo(i) = nan;
                cod(i) = nan;
            end
            i = i + 1;
        end
        ustar = ustar';
        zo = zo';
        cod = cod';
    else % Fill with nans if not computing (turn off pending more testing--PRJ 6-30-11)
        ustar = nan.*ones(size(EnsNo));
        zo = nan.*ones(size(EnsNo));
    end
    
    %Clear the structure
    clear A 
    
    %Save the data
    
    outmat = [EnsNo MTyear MTmon MTday MThour MTmin MTsec latlon Heading Pitch Roll Temp Depth BeamDepths DABack DAVeast DAVnorth DAVmag DAVdir DAVvert ustar zo]; 
    
    % Replace nans with -9999 (ARCGIS takes nans to be zero when exporting to
    % shapefile)
    if 0  % Fill the nans  
        for i = 7:size(outmat,2)
            outmat(:,i) = inpaint_nans(outmat(:,i));
        end 
    else  %fill with -9999
        outmat(isnan(outmat)) = -9999;
    end
    
   
    if append_data & zi == 1
        outfile = [fileName(1:end-4) '_GIS.csv'];
        firstfile = outfile;
    elseif ~append_data
        outfile = [fileName(1:end-4) '_GIS.csv'];
    elseif append_data & zi > 1
        outfile = firstfile;
    end    
            
    
    
    if append_data & zi == 1
        ofid   = fopen(outfile, 'wt');
        outcount = fprintf(ofid,'EnsNo, Year, Month, Day, Hour, Min, Sec, Lat_WGS84, Lon_WGS84, Heading_deg,  Pitch_deg,  Roll_deg, Temp_C, Depth_m, B1Depth_m, B2Depth_m, B3Depth_m, B4Depth_m, BackScatter_db, DAVeast_cmps, DAVnorth_cmps, DAVmag_cmps, DAVdir_deg, DAVvert_cmps, U_star_mps, Z0_m\n');
    elseif ~append_data
        ofid   = fopen(outfile, 'wt');
        outcount = fprintf(ofid,'EnsNo, Year, Month, Day, Hour, Min, Sec, Lat_WGS84, Lon_WGS84, Heading_deg,  Pitch_deg,  Roll_deg, Temp_C, Depth_m, B1Depth_m, B2Depth_m, B3Depth_m, B4Depth_m, BackScatter_db, DAVeast_cmps, DAVnorth_cmps, DAVmag_cmps, DAVdir_deg, DAVvert_cmps, U_star_mps, Z0_m\n');
    elseif append_data & zi > 1
        ofid   = fopen(outfile, 'at');
    end
    outcount = fprintf(ofid,'%6.0f, %4.0f, %2.0f, %2.0f, %2.0f, %2.0f, %2.2f, %3.10f, %3.10f, %3.3f, %3.3f, %3.3f, %3.1f, %3.2f, %3.2f, %3.2f, %3.2f, %3.2f, %3.0f, %3.2f, %3.2f, %3.2f, %3.1f, %3.2f, %2.4f, %2.4f\n',outmat');
    fclose(ofid);
    
    [Easting,Northing,utmzone] = deg2utm(latlon(:,1),latlon(:,2));
    
    VelOut = [VelOut; Easting Northing zeros(size(Easting)) DAVeast./100 DAVnorth./100];
       
    clear outmat outfile latlon EnsNo MTyear MTmon MTday MThour MTmin MTsec latlon Heading Pitch Roll Temp Depth BeamDepths DABack DAVeast DAVnorth DAVmag DAVdir DAVvert ustar zo Easting Northing utmzone
end

msgbox('Conversion Complete','Conversion Status','help','replace');

% Save an *.anv file (for iRIC model interface)
goodrows = [];
for i = 1:length(VelOut)
    rowsum = sum(isnan(VelOut(i,:)));
    if rowsum == 0
        goodrows = [goodrows; i];
    end
end
outfile = [fileName(1:end-4) '_DAV.anv'];
ofid    = fopen(outfile, 'wt');
outcount = fprintf(ofid,'%8.2f  %8.2f  %5.2f  %3.3f  %3.3f\n',VelOut(goodrows,:)');
fclose(ofid);


% Save the paths
if exist('LastDir.mat') == 2
    save('LastDir.mat','ascii2gispath','-append')
else
    save('LastDir.mat','ascii2gispath')
end



