function [exyz]=depthxyz(depthRaw,draft,pitchRaw,roll,heading,beamAngle,...
    unitsID,x,y,elev,ens);
% depthxyz computes the x,y, and z components of the location where each
% beam reflects from the streambed using the algorithm provided by Gary
% Murdock, RDI, 10-25-2002
%
% INPUT
% depthRaw - matrix of beam depths
% draft - draft of instrument
% pitchRaw - pitch vector from ADCP in degrees
% roll - roll vector from ADCP in degrees
% heading - heading vector from ADCP in degrees
% beamAngle - beam angle of instrument in degrees
% magvar - magnetic variation for site
% unitsID - units identifier
% x - x-coordinate(Easting) of center of ADCP
% y - y-coordinate(Northing) of center of ADCP
% elev - elevation of water-surface at ADCP
% ens - vector of ensemble numbers
%
% OUTPUT
%
% exyz - matrix with rows of ensembles and columns of x,y, and z
%
% David S. Mueller
% U.S. Geological Survey
% Office of Surface Water
% dmueller@usgs.gov
% 
% June 29, 2006

    % Correct draft for units
    if unitsID=='ft'
            draft=draft*0.0328083;
        else
            draft=draft*.01;
    end;
        
    % Create geo matrix of x, y, and elevation of transducers    
    geo=[x,y,-1.*repmat(draft,size(y))+elev];    

    % Compute slant range of each beam
    beamAngleR=beamAngle.*pi/180;    
    range=(depthRaw-draft)./cos(beamAngleR);
    
    % Adjust heading, pitch, and roll
    rollR=roll.*pi/180;
    pitchRawR=pitchRaw.*pi/180;
    headingR=heading.*pi/180;
    pitchR=atan(tan(pitchRawR).*cos(rollR));

    % Compute sine and cosine values
    ch=cos(headingR);
    sh=sin(headingR);
    cp=cos(pitchR);
    sp=sin(pitchR);
    cr=cos(rollR);
    sr=sin(rollR);
    
    % Configure transformation vectors for east, north, and vertical
    trans2e=[ch.*cr+sh.*sp.*sr sh.*cp ch.*sr-sh.*sp.*cr];
    trans2n=[-sh.*cr+ch.*sp.*sr ch.*cp -sh.*sr-ch.*sp.*cr];
    trans2u=[-cp.*sr sp cp.*cr];
    
    % Create matrix to convert from slant range to xyz
    rng2xyz=[-sin(beamAngleR) 0 -cos(beamAngleR);...
              sin(beamAngleR) 0 -cos(beamAngleR);...
              0 sin(beamAngleR) -cos(beamAngleR);...
             0 -sin(beamAngleR) -cos(beamAngleR)];
    
    % Compute xyz for each beam
    beam1xyz=repmat(rng2xyz(1,:),size(range,1),1).*repmat(range(:,1),1,3);
    beam2xyz=repmat(rng2xyz(2,:),size(range,1),1).*repmat(range(:,2),1,3);
    beam3xyz=repmat(rng2xyz(3,:),size(range,1),1).*repmat(range(:,3),1,3);
    beam4xyz=repmat(rng2xyz(4,:),size(range,1),1).*repmat(range(:,4),1,3);
    
    % Correct beam 1 xyz for heading, pitch, and roll and compute elevation
    beam1enu(:,1)=ens;
    beam1enu(:,2)=sum(trans2e.*beam1xyz,2);
    beam1enu(:,3)=sum(trans2n.*beam1xyz,2);
    beam1enu(:,4)=sum(trans2u.*beam1xyz,2);
    beam1enu(:,2:4)=beam1enu(:,2:4)+geo;
    
    % Correct beam 2 xyz for heading, pitch, and roll and compute elevation
    beam2enu(:,1)=ens;
    beam2enu(:,2)=sum(trans2e.*beam2xyz,2);
    beam2enu(:,3)=sum(trans2n.*beam2xyz,2);
    beam2enu(:,4)=sum(trans2u.*beam2xyz,2);
    beam2enu(:,2:4)=beam2enu(:,2:4)+geo;
    
    % Correct beam 3 xyz for heading, pitch, and roll and compute elevation
    beam3enu(:,1)=ens;
    beam3enu(:,2)=sum(trans2e.*beam3xyz,2);
    beam3enu(:,3)=sum(trans2n.*beam3xyz,2);
    beam3enu(:,4)=sum(trans2u.*beam3xyz,2);
    beam3enu(:,2:4)=beam3enu(:,2:4)+geo;
    
    % Correct beam 4 xyz for heading, pitch, and roll and compute elevation
    beam4enu(:,1)=ens;
    beam4enu(:,2)=sum(trans2e.*beam4xyz,2);
    beam4enu(:,3)=sum(trans2n.*beam4xyz,2);
    beam4enu(:,4)=sum(trans2u.*beam4xyz,2);
    beam4enu(:,2:4)=beam4enu(:,2:4)+geo;
    
    % Create final matrix
    exyz=[beam1enu;beam2enu;beam3enu;beam4enu];
    exyz=sortrows(exyz,1);

